import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Calendar, MapPin, Clock, User, Star, Award, Navigation, Building2, UtensilsCrossed, TreePine, Camera, CheckCircle, ArrowRight, Route, Sparkles, Zap } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './BookTour.css'

const BookTour = () => {
  const navigate = useNavigate()
  const [selectedTheme, setSelectedTheme] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)
  const [currentStep, setCurrentStep] = useState(1)

  const themes = [
    { 
      id: 'historical', 
      name: 'Историческая', 
      icon: Building2,
      color: '#2E86AB',
      gradient: 'linear-gradient(135deg, #2E86AB 0%, #1E5F7A 100%)',
      description: 'Погрузитесь в многовековую историю города',
      duration: '3 часа',
      price: 'от 1500₽',
      features: ['Эксклюзивные маршруты', 'Исторические факты', 'Архитектурные детали']
    },
    { 
      id: 'gastronomic', 
      name: 'Гастрономическая', 
      icon: UtensilsCrossed,
      color: '#A23B72',
      gradient: 'linear-gradient(135deg, #A23B72 0%, #7A2B55 100%)',
      description: 'Откройте вкусы и традиции местной кухни',
      duration: '2.5 часа',
      price: 'от 2000₽',
      features: ['Дегустации', 'Исторические рецепты', 'Встречи с шеф-поварами']
    },
    { 
      id: 'nature', 
      name: 'Природная', 
      icon: TreePine,
      color: '#06A77D',
      gradient: 'linear-gradient(135deg, #06A77D 0%, #048A68 100%)',
      description: 'Насладитесь красотой природы в черте города',
      duration: '4 часа',
      price: 'от 1800₽',
      features: ['Экологические тропы', 'Наблюдение за природой', 'Фотосессии']
    },
    { 
      id: 'architectural', 
      name: 'Архитектурная', 
      icon: Camera,
      color: '#F18F01',
      gradient: 'linear-gradient(135deg, #F18F01 0%, #D67E00 100%)',
      description: 'Изучите уникальную архитектуру разных эпох',
      duration: '3.5 часа',
      price: 'от 1700₽',
      features: ['Архитектурные стили', 'Секреты строительства', 'Современные проекты']
    }
  ]

  const guides = [
    { name: 'Анна Петрова', theme: 'Историческая', experience: '8 лет', rating: 5, tours: '1200+', specialty: 'Средневековая история' },
    { name: 'Михаил Соколов', theme: 'Гастрономическая', experience: '6 лет', rating: 5, tours: '800+', specialty: 'Традиционная кухня' },
    { name: 'Елена Волкова', theme: 'Природная', experience: '10 лет', rating: 5, tours: '1500+', specialty: 'Экология и природа' },
    { name: 'Дмитрий Новиков', theme: 'Архитектурная', experience: '7 лет', rating: 5, tours: '950+', specialty: 'Современная архитектура' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedTheme && selectedDate) {
      const savedTours = localStorage.getItem('excursionsTours')
      const tours = savedTours ? JSON.parse(savedTours) : []
      
      const themeData = themes.find(t => t.id === selectedTheme)
      const guideData = guides.find(g => g.theme === themeData?.name) || guides[0]
      
      const newTour = {
        id: Date.now(),
        route: themeData?.name || 'Экскурсия',
        guide: guideData.name,
        date: selectedDate,
        rating: 0,
        duration: themeData?.duration || '3 часа',
        theme: themeData?.name || 'Историческая',
        highlights: themeData?.features || []
      }
      
      tours.push(newTour)
      localStorage.setItem('excursionsTours', JSON.stringify(tours))
      
      setIsSubmitted(true)
      setTimeout(() => {
        navigate('/')
      }, 2000)
    }
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  return (
    <div className="book-tour">
      <div className="container">
        <motion.section 
          className="book-hero-immersive"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-content-book">
            <motion.div 
              className="hero-badge-book"
              initial={{ scale: 0, rotate: -180 }}
              animate={{ scale: 1, rotate: 0 }}
              transition={{ delay: 0.2, type: "spring", stiffness: 200 }}
            >
              <Navigation size={20} strokeWidth={2.5} />
              <span>ЗАПИСАТЬСЯ НА ЭКСКУРСИЮ</span>
            </motion.div>
            <h1 className="hero-title-book">
              Выберите <span className="title-highlight-book">тематику</span>
            </h1>
            <p className="hero-description-book">
              Откройте для себя удивительные маршруты. Каждая экскурсия — это 
              уникальное путешествие в историю, культуру и природу города.
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="book-form-immersive"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.3 }}
        >
          <div className="form-progress">
            <div className="progress-steps">
              <div className={`progress-step ${currentStep >= 1 ? 'active' : ''}`}>
                <div className="step-number">1</div>
                <span>Тема</span>
              </div>
              <div className="progress-line"></div>
              <div className={`progress-step ${currentStep >= 2 ? 'active' : ''}`}>
                <div className="step-number">2</div>
                <span>Дата</span>
              </div>
              <div className="progress-line"></div>
              <div className={`progress-step ${currentStep >= 3 ? 'active' : ''}`}>
                <div className="step-number">3</div>
                <span>Подтверждение</span>
              </div>
            </div>
          </div>

          <form className="book-form-wrapper-immersive" onSubmit={handleSubmit}>
            <div className="form-section-theme">
              <div className="form-label-immersive">
                <MapPin size={28} strokeWidth={2.5} />
                <span className="label-text-immersive">Тема экскурсии</span>
              </div>
              <div className="themes-wheel">
                {themes.map((theme, index) => {
                  const Icon = theme.icon
                  const isSelected = selectedTheme === theme.id
                  return (
                    <motion.button
                      key={theme.id}
                      type="button"
                      className={`theme-wheel-card ${isSelected ? 'selected' : ''}`}
                      onClick={() => {
                        setSelectedTheme(theme.id)
                        setCurrentStep(2)
                      }}
                      initial={{ opacity: 0, rotateY: -90 }}
                      animate={{ opacity: 1, rotateY: 0 }}
                      transition={{ delay: index * 0.1 }}
                      whileHover={{ scale: 1.05, rotate: 2 }}
                      whileTap={{ scale: 0.95 }}
                      style={isSelected ? { 
                        background: theme.gradient,
                        borderColor: theme.color 
                      } : {}}
                    >
                      <div className="theme-wheel-icon" style={{ 
                        background: isSelected ? 'rgba(255, 255, 255, 0.2)' : 'rgba(46, 134, 171, 0.1)',
                        borderColor: isSelected ? 'rgba(255, 255, 255, 0.4)' : theme.color
                      }}>
                        <Icon size={56} strokeWidth={2} style={{ color: isSelected ? 'white' : theme.color }} />
                        {isSelected && (
                          <motion.div 
                            className="icon-glow"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            transition={{ type: "spring", stiffness: 300 }}
                          />
                        )}
                      </div>
                      <div className="theme-wheel-content">
                        <h3 className="theme-wheel-name" style={{ color: isSelected ? 'white' : 'var(--text)' }}>
                          {theme.name}
                        </h3>
                        <p className="theme-wheel-description" style={{ color: isSelected ? 'rgba(255, 255, 255, 0.9)' : 'var(--text-light)' }}>
                          {theme.description}
                        </p>
                        <div className="theme-wheel-features">
                          {theme.features.map((feature, i) => (
                            <div key={i} className="feature-tag" style={{ 
                              background: isSelected ? 'rgba(255, 255, 255, 0.2)' : 'rgba(46, 134, 171, 0.08)',
                              color: isSelected ? 'white' : 'var(--primary)'
                            }}>
                              {feature}
                            </div>
                          ))}
                        </div>
                        <div className="theme-wheel-meta">
                          <div className="meta-item">
                            <Clock size={16} strokeWidth={2} />
                            <span>{theme.duration}</span>
                          </div>
                          <div className="meta-item">
                            <span>{theme.price}</span>
                          </div>
                        </div>
                      </div>
                      {isSelected && (
                        <motion.div 
                          className="theme-check-wheel"
                          initial={{ scale: 0, rotate: -180 }}
                          animate={{ scale: 1, rotate: 0 }}
                          transition={{ type: "spring", stiffness: 300 }}
                        >
                          <CheckCircle size={36} strokeWidth={2.5} />
                        </motion.div>
                      )}
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <AnimatePresence>
              {selectedTheme && (
                <motion.div 
                  className="form-section-date"
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  exit={{ opacity: 0, height: 0 }}
                  transition={{ duration: 0.4 }}
                  onAnimationComplete={() => setCurrentStep(3)}
                >
                  <div className="date-label-immersive">
                    <Calendar size={28} strokeWidth={2.5} />
                    <div className="date-label-content-immersive">
                      <span className="date-label-title-immersive">Дата экскурсии</span>
                      <span className="date-label-hint-immersive">Выберите удобную дату из доступных</span>
                    </div>
                  </div>
                  <div className="date-picker-wrapper">
                    <input
                      type="date"
                      value={selectedDate}
                      onChange={(e) => setSelectedDate(e.target.value)}
                      min={minDateStr}
                      max={maxDateStr}
                      className="date-picker-immersive"
                      required
                    />
                    <div className="date-picker-accent"></div>
                  </div>
                </motion.div>
              )}
            </AnimatePresence>

            <motion.button 
              type="submit" 
              className="submit-button-immersive"
              disabled={!selectedTheme || !selectedDate}
              whileHover={{ scale: 1.02 }}
              whileTap={{ scale: 0.98 }}
            >
              <span>Записаться на экскурсию</span>
              <ArrowRight size={24} strokeWidth={2.5} />
              <div className="button-shine"></div>
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message-immersive"
                initial={{ opacity: 0, scale: 0.8 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ type: "spring", stiffness: 300 }}
              >
                <CheckCircle size={48} strokeWidth={2.5} />
                <span>Вы успешно записаны на экскурсию!</span>
                <Sparkles size={24} strokeWidth={2} />
              </motion.div>
            )}
          </form>
        </motion.section>

        <ImageSlider />

        <motion.section 
          className="guides-immersive"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.4 }}
        >
          <div className="section-header-immersive">
            <div className="section-line"></div>
            <h2 className="section-title-immersive">Наши гиды</h2>
            <div className="section-line"></div>
          </div>
          <div className="guides-showcase">
            {guides.map((guide, index) => (
              <motion.div
                key={index}
                className="guide-showcase-card"
                initial={{ opacity: 0, y: 30 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                whileHover={{ scale: 1.05, y: -8 }}
              >
                <div className="guide-avatar-showcase" style={{ background: 'var(--gradient-1)' }}>
                  <User size={36} strokeWidth={2} />
                  <div className="avatar-ring"></div>
                </div>
                <h3 className="guide-name-showcase">{guide.name}</h3>
                <div className="guide-specialty">{guide.specialty}</div>
                <div className="guide-stats">
                  <div className="guide-stat">
                    <Clock size={16} strokeWidth={2} />
                    <span>{guide.experience}</span>
                  </div>
                  <div className="guide-stat">
                    <Star size={16} strokeWidth={2} fill="#F18F01" color="#F18F01" />
                    <span>{guide.rating}.0</span>
                  </div>
                  <div className="guide-stat">
                    <Award size={16} strokeWidth={2} />
                    <span>{guide.tours}</span>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="why-section"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8, delay: 0.5 }}
        >
          <div className="section-header-immersive">
            <div className="section-line"></div>
            <h2 className="section-title-immersive">Почему выбирают нас</h2>
            <div className="section-line"></div>
          </div>
          <div className="why-grid">
            <motion.div 
              className="why-card"
              whileHover={{ rotate: 1, scale: 1.03 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="why-icon" style={{ background: 'var(--gradient-1)' }}>
                <Zap size={32} strokeWidth={2} />
              </div>
              <h3 className="why-title">Эксклюзивные маршруты</h3>
              <p className="why-text">
                Только у нас вы найдете уникальные маршруты, разработанные 
                нашими экспертами и недоступные в обычных турах.
              </p>
            </motion.div>
            <motion.div 
              className="why-card"
              whileHover={{ rotate: -1, scale: 1.03 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="why-icon" style={{ background: 'var(--gradient-2)' }}>
                <Route size={32} strokeWidth={2} />
              </div>
              <h3 className="why-title">Интерактивные туры</h3>
              <p className="why-text">
                Не просто слушайте, а активно участвуйте! Интерактивные элементы 
                делают каждую экскурсию незабываемым приключением.
              </p>
            </motion.div>
            <motion.div 
              className="why-card"
              whileHover={{ rotate: 1, scale: 1.03 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="why-icon" style={{ background: 'var(--gradient-3)' }}>
                <Award size={32} strokeWidth={2} />
              </div>
              <h3 className="why-title">Профессиональные гиды</h3>
              <p className="why-text">
                Наши гиды — это не просто экскурсоводы, а настоящие энтузиасты, 
                готовые поделиться своими знаниями и страстью.
              </p>
            </motion.div>
            <motion.div 
              className="why-card"
              whileHover={{ rotate: -1, scale: 1.03 }}
              transition={{ type: "spring", stiffness: 300 }}
            >
              <div className="why-icon" style={{ background: 'var(--gradient-1)' }}>
                <Sparkles size={32} strokeWidth={2} />
              </div>
              <h3 className="why-title">Незабываемые впечатления</h3>
              <p className="why-text">
                Каждая экскурсия — это не просто прогулка, а настоящее путешествие 
                во времени, пространстве и культуре.
              </p>
            </motion.div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookTour
